/**
 * Renderer Process Logger with IPC Integration
 * Author: Dario Pascoal
 *
 * Description: This module provides a comprehensive logging system for the renderer
 * process in the DHL All-in-one Electron application. It bridges browser-based
 * JavaScript code with the main process logging infrastructure through IPC
 * (Inter-Process Communication) channels.
 *
 * Key Functionality:
 * - Cross-process logging from renderer to main process
 * - Automatic source file detection for better debugging
 * - Graceful fallback to console logging when IPC fails
 * - Support for all standard log levels (info, debug, warn, error)
 * - Error object handling with stack trace extraction
 * - Log file path retrieval for external access
 *
 * Technical Implementation:
 * The logger uses Electron's IPC mechanism to send log messages from the renderer
 * process to the main process, where they are written to persistent log files.
 * This approach ensures that all application logs are centralized while maintaining
 * the ability to debug renderer-specific issues.
 *
 * Business Context:
 * In production environments, having centralized logging is critical for
 * troubleshooting user issues and monitoring application health. This logger
 * ensures that frontend JavaScript errors and information are captured alongside
 * backend Node.js logs in a unified logging system.
 *
 * Integration Points:
 * - Uses window.electron API exposed by the preload script
 * - Communicates with main process logger via IPC channels
 * - Integrates with the application's overall logging infrastructure
 * - Provides consistent logging interface across all renderer processes
 */

// Import the Electron IPC interface exposed by the preload script
// This provides secure access to IPC functionality from the renderer process
const ipcRenderer = window.electron;

// ================================================================================================
// LOGGER CLASS - RENDERER PROCESS LOGGING IMPLEMENTATION
// ================================================================================================

/**
 * Logger Class for Renderer Process
 *
 * This class provides a comprehensive logging solution specifically designed for
 * Electron renderer processes. It handles the complex task of sending log messages
 * from browser-based JavaScript code to the main process for persistent storage.
 *
 * Key Features:
 * - Automatic source file detection using stack trace analysis
 * - IPC-based communication with main process logger
 * - Graceful fallback to console logging when IPC fails
 * - Support for error objects with stack trace extraction
 * - Singleton pattern for consistent logging across the application
 *
 * Architecture Benefits:
 * - Centralized log management in the main process
 * - Consistent logging interface across all renderer windows
 * - Automatic error handling and fallback mechanisms
 * - Source file tracking for better debugging capabilities
 */
class Logger {
  /**
   * Initialize the Logger instance
   *
   * Sets up the logger with a ready state and provides confirmation that
   * the logging system is available. The logger is immediately ready to
   * accept log messages upon instantiation.
   */
  constructor() {
    this.ready = true;
    console.log("Logger instance created");
  }

  // ================================================================================================
  // SOURCE FILE DETECTION AND DEBUGGING UTILITIES
  // ================================================================================================

  /**
   * Extract the source file name from the current call stack
   *
   * This method uses JavaScript's stack trace functionality to determine which
   * file initiated the logging call. This information is invaluable for debugging
   * as it automatically identifies the source of log messages without requiring
   * manual source identification.
   *
   * Technical Implementation:
   * - Creates an Error object to access the call stack
   * - Parses the stack trace to find the calling file
   * - Handles both forward slash (Unix) and backslash (Windows) path separators
   * - Extracts just the filename without path or line number information
   *
   * Fallback Strategy:
   * If stack parsing fails (which can happen in minified code or certain
   * execution contexts), the method returns "RENDERER" as a generic identifier
   * to ensure logging continues to function.
   *
   * @returns {string} The filename of the calling source or "RENDERER" as fallback
   */
  getSource() {
    try {
      // Generate a stack trace to analyze the call hierarchy
      const stack = new Error().stack;

      // Extract the third line of the stack (index 3) which represents the actual caller
      // Stack trace format: Error\n  at getSource\n  at logMethod\n  at actualCaller
      const callerFile = stack.split("\n")[3].trim().split(" ")[1];

      // Parse the file path to extract just the filename
      // Handle both Unix (/) and Windows (\) path separators
      // Remove line number and column information (everything after :)
      const fileName = callerFile
        .split("/") // Split Unix paths
        .pop() // Get last segment
        .split("\\") // Split Windows paths
        .pop() // Get last segment
        .split(":")[0]; // Remove line/column info

      return fileName || "RENDERER";
    } catch (e) {
      // Fallback to generic identifier if stack parsing fails
      // This can happen in minified code or certain execution environments
      return "RENDERER";
    }
  }

  // ================================================================================================
  // STANDARD LOGGING METHODS WITH IPC INTEGRATION
  // ================================================================================================

  /**
   * Log informational messages
   *
   * This method handles general informational logging that helps track application
   * flow and user actions. Info messages are typically used for significant events
   * that are part of normal operation.
   *
   * Dual Output Strategy:
   * 1. Immediate console output for development debugging
   * 2. IPC transmission to main process for persistent file logging
   *
   * Error Handling:
   * The method includes comprehensive error handling to ensure that logging
   * failures don't interrupt application flow. If IPC communication fails,
   * the console output still provides immediate debugging information.
   *
   * @param {string} message - The informational message to log
   */
  info(message) {
    // Immediate console output for development and debugging
    console.info(message);

    try {
      // Attempt to send the message to the main process for persistent logging
      ipcRenderer
        .invoke("log-message", "info", this.getSource(), message)
        .catch(() => {
          // Silent fallback - the console.info above already provided output
          // We don't want to spam the console with IPC errors
        });
    } catch (err) {
      // Silent fallback - already logged to console above
      // This catch handles cases where ipcRenderer is not available
    }
  }

  /**
   * Log debug messages for development and troubleshooting
   *
   * Debug messages provide detailed information about application state and
   * execution flow. These are typically used during development and can be
   * filtered out in production environments if needed.
   *
   * @param {string} message - The debug message to log
   */
  debug(message) {
    // Immediate console output for development visibility
    console.debug(message);

    try {
      // Send debug information to main process for comprehensive logging
      ipcRenderer
        .invoke("log-message", "debug", this.getSource(), message)
        .catch(() => {
          // Silent fallback - console output already provided
        });
    } catch (err) {
      // Silent fallback - already logged to console
    }
  }

  /**
   * Log warning messages for potential issues
   *
   * Warning messages indicate situations that are unusual but not necessarily
   * errors. They help identify potential problems before they become critical
   * issues and are important for proactive monitoring.
   *
   * @param {string} message - The warning message to log
   */
  warn(message) {
    // Immediate console warning for developer attention
    console.warn(message);

    try {
      // Transmit warning to main process for monitoring and analysis
      ipcRenderer
        .invoke("log-message", "warn", this.getSource(), message)
        .catch(() => {
          // Silent fallback - console warning already displayed
        });
    } catch (err) {
      // Silent fallback - already logged to console
    }
  }

  /**
   * Log error messages with enhanced error object handling
   *
   * This method provides comprehensive error logging that can handle both simple
   * error messages and complex Error objects with stack traces. It's designed to
   * capture maximum debugging information while maintaining clean console output.
   *
   * Error Processing Strategy:
   * - Simple strings are logged as-is
   * - Error objects are processed to extract message and stack trace
   * - Combined messages include both user message and error details
   * - Stack traces are preserved for debugging purposes
   *
   * Business Context:
   * Error logging is critical for production support and debugging. This method
   * ensures that all error information is captured comprehensively, enabling
   * effective troubleshooting of issues reported by users.
   *
   * @param {string} message - The error message describing the context
   * @param {Error|string|null} error - Optional error object or additional details
   */
  error(message, error = null) {
    let errorMsg = message;

    // Process error parameter if provided
    if (error) {
      // Log to console with both message and error for immediate debugging
      console.error(message, error);

      if (error instanceof Error) {
        // Extract comprehensive information from Error objects
        // Include both the error message and stack trace for debugging
        errorMsg += `: ${error.message}\n${error.stack || ""}`;
      } else {
        // Handle non-Error objects by converting to string representation
        errorMsg += `: ${String(error)}`;
      }
    } else {
      // Simple message-only error logging
      console.error(message);
    }

    try {
      // Send the comprehensive error message to main process for persistent logging
      ipcRenderer
        .invoke("log-message", "error", this.getSource(), errorMsg)
        .catch(() => {
          // Silent fallback - console error already displayed
        });
    } catch (err) {
      // Silent fallback - already logged to console above
    }
  }

  // ================================================================================================
  // LOG FILE ACCESS AND UTILITY METHODS
  // ================================================================================================

  /**
   * Retrieve the current log file path from the main process
   *
   * This method provides access to the file system path where log messages are
   * being stored. This is useful for external tools, support diagnostics, or
   * allowing users to access their log files for troubleshooting.
   *
   * Business Context:
   * Support teams often need access to log files to diagnose user issues.
   * This method enables the application to provide users with the exact
   * location of their log files, facilitating better support experiences.
   *
   * Technical Implementation:
   * - Uses IPC to request the log path from the main process
   * - Returns null if the path cannot be retrieved (graceful degradation)
   * - Handles both IPC failures and main process errors transparently
   *
   * @returns {Promise<string|null>} The absolute path to the log file, or null if unavailable
   */
  async getLogPath() {
    try {
      // Request the current log file path from the main process
      return await ipcRenderer.invoke("get-log-path").catch(() => null);
    } catch (err) {
      // Log the error for debugging but don't throw - return null for graceful handling
      console.error("Failed to get log path:", err);
      return null;
    }
  }
}

// ================================================================================================
// SINGLETON EXPORT FOR CONSISTENT LOGGING
// ================================================================================================

/**
 * SINGLETON LOGGER INSTANCE EXPORT
 *
 * The Logger class is exported as a singleton instance to ensure consistent
 * logging behavior across all parts of the renderer process. This approach
 * provides several important benefits:
 *
 * Consistency Benefits:
 * - Single point of configuration for all logging in the renderer
 * - Consistent source detection and IPC communication
 * - Unified error handling and fallback mechanisms
 * - Prevents multiple logger instances with different states
 *
 * Memory Efficiency:
 * - Reduces memory overhead by avoiding duplicate logger instances
 * - Shares the same IPC connection across all logging calls
 * - Maintains consistent state without redundant initialization
 *
 * Integration Simplicity:
 * - Modules can simply require this file and start logging immediately
 * - No need to manage logger lifecycle or configuration
 * - Consistent interface across all renderer windows and contexts
 *
 * Usage throughout the application:
 * const logger = require('./log-helper');
 * logger.info('Application started');
 * logger.error('Something went wrong', errorObject);
 */
module.exports = new Logger();
